/* Copyright (C) 2002-2018 RealVNC Ltd.  All Rights Reserved.
*/

#ifndef UUID_73f19da2_1152_45ec_9a5e_5ff8dcbc8ede
#define UUID_73f19da2_1152_45ec_9a5e_5ff8dcbc8ede

/**
 * \file vncint.h
 *
 * This header defines integer types used by several of the VNC SDKs.
 *
 * Before including this header, or another header that includes this one, you
 * must instruct the SDK how to define these types.  You must define exactly
 * one of:
 *
 *  - VNC_USE_STDINT_H - Use this if your build environment defines the C99
 *    header stdint.h, or you can provide an equivalent.
 *
 *  - VNC_USE_DEFAULT_INT_TYPEDEFS - Use this to provide default definitions
 *    for these types.  The default definitions assume that the following are
 *    true:
 *     - sizeof(short) == 2
 *     - sizeof(long) == 4
 *     - sizeof(long long) == 8
 *
 * Using VNC_USE_DEFAULT_INT_TYPEDEFS is sufficient for many uses, but
 * use of VNC_USE_STDINT_H is preferred.
 *
 * Make sure to use the VNCVerifyIntegerTypes() macro to check whether these
 * types have been defined correctly.
 *
 * By default, this header will also define vnc_float_t and vnc_double_t. These
 * types must be respectively capable of storing a IEE754-1985 single- or
 * double-precision floating point number. The default definitions are for
 * float and double, which should work with any C99-compliant compiler.
 *
 * If your compiler's float and double types are not suitable for this purpose,
 * then you can suppress these definitions by defining
 * VNC_DISABLE_FLOAT_TYPEDEFS. In that case, if your application calls any APIs
 * that use vnc_float_t or vnc_double_t in their parameter list, then you must
 * define these types yourself.
 *
 * \see VNCVerifyIntegerTypes, vnc_int8_t, vnc_int16_t, vnc_int32_t,
 * vnc_int64_t, vnc_uint8_t, vnc_uint16_t, vnc_uint32_t, vnc_uint64_t,
 * vnc_intptr_t, vnc_uintptr_t, vnc_float_t, vnc_double_t
 */

#ifdef __cplusplus
extern "C"
{
#endif

#include <stddef.h>

#ifdef __SYMBIAN32__
    #include <e32def.h>
    typedef TInt8 vnc_int8_t;
    typedef TInt16 vnc_int16_t;
    typedef TInt32 vnc_int32_t;
    typedef TInt64 vnc_int64_t;
    typedef TUint8 vnc_uint8_t;
    typedef TUint16 vnc_uint16_t;
    typedef TUint32 vnc_uint32_t;
    typedef TUint64 vnc_uint64_t;
    typedef TAny vnc_void_t;
    typedef vnc_uint32_t vnc_size_t;
#else
#ifdef VNC_USE_STDINT_H
    #ifdef VNC_USE_DEFAULT_INT_TYPEDEFS
        #error you must define exactly one of VNC_USE_STDINT_H or VNC_USE_DEFAULT_INT_TYPEDEFS
    #else
        #include <stdint.h>
        typedef int8_t vnc_int8_t;
        typedef int16_t vnc_int16_t;
        typedef int32_t vnc_int32_t;
        typedef int64_t vnc_int64_t;
        typedef uint8_t vnc_uint8_t;
        typedef uint16_t vnc_uint16_t;
        typedef uint32_t vnc_uint32_t;
        typedef uint64_t vnc_uint64_t;
        typedef void vnc_void_t;
        typedef size_t vnc_size_t;
    #endif
#else
    #ifdef VNC_USE_DEFAULT_INT_TYPEDEFS
        typedef signed char vnc_int8_t;
        typedef signed short vnc_int16_t;
        typedef signed long vnc_int32_t;
        typedef unsigned char vnc_uint8_t;
        typedef unsigned short vnc_uint16_t;
        typedef unsigned long vnc_uint32_t;
        typedef void vnc_void_t;
        typedef size_t vnc_size_t;

        #ifdef _MSC_VER 
          typedef signed __int64 vnc_int64_t;
          typedef unsigned __int64 vnc_uint64_t;
        #else
          typedef signed long long vnc_int64_t;
          typedef unsigned long long vnc_uint64_t;
        #endif
    #else
        #error you must define exactly one of VNC_USE_STDINT_H or VNC_USE_DEFAULT_INT_TYPEDEFS
    #endif
#endif
#endif

/* If possible, use stdint's definition of intptr_t and uintptr_t. If not, work out
 * the appropriate data types to use for this platform.
 */
#if defined(VNC_USE_STDINT_H) && defined(__STDC_VERSION__) && (__STDC_VERSION__ >= 199901L)
    #include <inttypes.h>
    /** VNC equivalent of PRIxPTR. */
    #define VNC_PRIxPTR PRIxPTR
    typedef intptr_t vnc_intptr_t;
    typedef uintptr_t vnc_uintptr_t;
#else
    /* This check autodetects building for x86_64 if the compiler is gcc or
     * if it is Microsoft Visual C++ 2005 or later.  If you are using a
     * different compiler to build for x86_64, or you are building for some
     * other 64-bit architecture, you should define VNC_FORCE_64_BIT before
     * including this header.
     */
    #if defined(_WIN64) || defined(__x86_64__) || defined(VNC_FORCE_64_BIT) || defined(__aarch64__)
        /** VNC equivalent of PRIxPTR. */
        #define VNC_PRIxPTR "llx"
        typedef vnc_int64_t vnc_intptr_t;
        typedef vnc_uint64_t vnc_uintptr_t;
    #else
        /** VNC equivalent of PRIxPTR. */
        #define VNC_PRIxPTR "lx"
        typedef vnc_int32_t vnc_intptr_t;
        typedef vnc_uint32_t vnc_uintptr_t;
    #endif
#endif

#ifndef VNC_DISABLE_FLOAT_TYPEDEFS
typedef float vnc_float_t;
typedef double vnc_double_t;
#endif

typedef vnc_uint64_t vnc_time_t;

typedef vnc_uint8_t vnc_bool_t;

/** Use for a 'true' boolean value with \ref vnc_bool_t */
#define vnc_true 1
/** Use for a 'false' boolean value with \ref vnc_bool_t */
#define vnc_false 0

/**
 * \typedef vnc_bool_t
 * Boolean value.
 */

/**
 * \typedef vnc_int8_t
 * 8-bit signed integer.
 */

/**
 * \typedef vnc_int16_t
 * 16-bit signed integer.
 */

/**
 * \typedef vnc_int32_t
 * 32-bit signed integer.
 */

/**
 * \typedef vnc_int64_t
 * 64-bit signed integer.
 */

/**
 * \typedef vnc_uint8_t
 * 8-bit unsigned integer.
 */

/**
 * \typedef vnc_uint16_t
 * 16-bit unsigned integer.
 */

/**
 * \typedef vnc_uint32_t
 * 32-bit unsigned integer.
 */

/**
 * \typedef vnc_uint64_t
 * 64-bit unsigned integer.
 */

/**
 * \typedef vnc_void_t
 * Void type.
 */

/**
 * \typedef vnc_size_t
 * Size type.
 */

/**
 * \typedef vnc_intptr_t
 * Signed integer large enough to hold a memory address on the machine on which
 * the SDK is running.
 */

/**
 * \typedef vnc_uintptr_t
 * Unsigned integer large enough to hold a memory address on the machine on
 * which the SDK is running.
 */

/**
 * \typedef vnc_float_t
 * Single-precision floating point number.
 */

/**
 * \typedef vnc_double_t
 * Double-precision floating point number.
 */

/**
 * \typedef vnc_time_t
 * Data type for holding time values.
 */


/**
 * \brief Allows the application to verify that its use of VNC_USE_STDINT_H or 
 * VNC_USE_DEFAULT_INT_TYPEDEFS is correct.
 *
 * This is a preprocessor macro that is used to validate at runtime that the
 * integer types governed by the preprocessor macros VNC_USE_STDINT_H and
 * VNC_USE_DEFAULT_INT_TYPEDEFS have been defined correctly.
 *
 * Because a mismatch in integer type sizes between your application and the
 * SDK binary is a severe programming error, this macro uses the assert() macro
 * from the C standard library header assert.h to perform its validation.
 * Therefore, to use this macro, you must either include assert.h or define
 * your own equivalent to assert().  If you are using assert() from assert.h,
 * then please note that the behavior of VNCVerifyIntegerTypes() is
 * (as with all other assertions) governed by the NDEBUG macro.
 *
 * It is recommended that you use this macro once, at application startup.
 */
#define VNCVerifyIntegerTypes() \
    do \
    { \
      assert(sizeof(vnc_int8_t) == 1); \
      assert(sizeof(vnc_int16_t) == 2); \
      assert(sizeof(vnc_int32_t) == 4); \
      assert(sizeof(vnc_int64_t) == 8); \
      assert(sizeof(vnc_intptr_t) == sizeof(void *)); \
      assert(sizeof(vnc_uint8_t) == 1); \
      assert(sizeof(vnc_uint16_t) == 2); \
      assert(sizeof(vnc_uint32_t) == 4); \
      assert(sizeof(vnc_uint64_t) == 8); \
      assert(sizeof(vnc_uintptr_t) == sizeof(void *)); \
    } while (0)

#ifdef __cplusplus
}
#endif

#endif /* !defined(UUID_73f19da2_1152_45ec_9a5e_5ff8dcbc8ede) */
